/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.

	$Id: pgpBER.c,v 1.34.2.2 2001/06/01 14:17:28 jdool Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pgpMem.h"
#include "pgpContext.h"
#include "pgpPFLPriv.h"
#include "pgpBERPriv.h"
#include "pgpLDAP.h"
#include "pgpEndianConversion.h"

	PGPError
pgpNewBERElement(
	PGPMemoryMgrRef		memMgr,
	PGPberElementRef *	ber )
{
	PGPError		err 	= kPGPError_NoErr;

	PGPValidateMemoryMgr( memMgr );
	PGPValidatePtr( ber );

	*ber = PGPNewData(
				memMgr, 
				sizeof( PGPberElement ), 
				kPGPMemoryMgrFlags_Clear );
	if( IsNull( *ber ) )
		return kPGPError_OutOfMemory;

	(*ber)->encoding = NULL;
	(*ber)->length = 0;
	(*ber)->nextInsertIndex = 0;
	(*ber)->nextReadIndex = 0;
	(*ber)->tag = 0;

	(*ber)->next.nextSeq = NULL;
	(*ber)->prev.prevSeq = NULL;

	(*ber)->memMgr = memMgr;

	return err;
}

	PGPError
PGPNewBERElement(
	PGPContextRef		context,
	PGPberElementRef *	ber )
{
	PGPError			err 	= kPGPError_NoErr;
	PGPberElementRef	newBer	= kInvalidPGPberElementRef;
	PGPMemoryMgrRef		memMgr	= kInvalidPGPMemoryMgrRef;

	PGPValidateContext( context );
	PGPValidatePtr( ber );

	memMgr = PGPPeekContextMemoryMgr( context );

	err = pgpNewBERElement( memMgr, &newBer ); CKERR;

	*ber = newBer;

error:
	return err;
}

/*
 * This frees the current PGPberElement and any sequences/sets/PGPberElements
 * pointed to by the current one.
 */
	PGPError
PGPFreeBERElement(
	PGPberElementRef	ber )
{
	PGPValidateBERElementRef( ber );

	if( ber->next.nextSeq )
		PGPFreeBERElement( ber->next.nextSeq->nextBER );
	if( ber->prev.prevSeq )
		ber->prev.prevSeq->nextBER = NULL;
	
	if( IsntNull( ber->encoding ) )
		(void) PGPFreeData( ber->encoding );

	if( IsntNull( ber->next.nextSeq ) )
		(void) PGPFreeData( ber->next.nextSeq );

	(void) PGPFreeData( ber );

    return kPGPError_NoErr;
}

/****************************************************************************
 * BER encoding routines													*
 ****************************************************************************/

	PGPError 
pgpBERGetTagEncoding(
	PGPUInt32	tag,
	PGPByte *	s, 
	PGPSize *	length)
{
	PGPUInt32		i			= 0;
	PGPUInt32		mask		= 0;
	PGPUInt32		netTag		= 0;

	PGPValidatePtr( length );
	PGPValidatePtr( s );

	for( i = sizeof( PGPUInt32 ) - 1; i > 0; i-- )
	{
		mask = 0xFF << ( i * 8 );
		if( tag & mask )
			break;
	}

	*length = i + 1;

	netTag = PGPHostToNetLong( tag );

	pgpCopyMemory( ( (PGPByte *) &netTag ) + sizeof( PGPUInt32 ) - *length, s, *length );

	return kPGPError_NoErr;
}

	PGPError
pgpBERGetLengthEncoding(
	PGPSize 	length,
	PGPByte *	s,
	PGPSize *	lengthOfLength )
{
	PGPValidatePtr( lengthOfLength );
	PGPValidatePtr( s );

	/* Use short form if length <= 127 */
	if( length <= 0x7F )
		*lengthOfLength = 1;

	/* else use long encoding */
	else if( length <= 0xFFFF )
		*lengthOfLength = 2;
	else if( length <= 0xFFFFFF )
		*lengthOfLength = 3;
	else /* if( length <= 0xFFFFFFFF ) */
		*lengthOfLength = 4;

	if( *lengthOfLength == 1 )
		s[0] = ( length & 0xFF );
	else
	{
		PGPByte		netLength[sizeof(PGPUInt32)];
		
		s[0] = ( *lengthOfLength & 0xFF ) | kPGPberMask_BigLength;
		PGPUInt32ToEndian( length, kPGPBigEndian, &netLength[0] );
		pgpCopyMemory( &netLength[sizeof(PGPUInt32) - *lengthOfLength], &s[1],
			*lengthOfLength );

		/* We need to encode the length of the length, too */
		(*lengthOfLength)++;
	}

	return kPGPError_NoErr;
}

	PGPError
pgpBERStrAppend(
	PGPberElementRef	ber,
	PGPByte *			src, 
	PGPUInt32			tag, 
	PGPSize 			srcLength )
{
	PGPByte 	*s					= NULL;
	PGPSize 	length				= 0;
	PGPSize 	lengthLength		= 0;
	PGPByte 	lengthEncoding[8];
	PGPSize 	tagLength			= 0;
	PGPByte 	tagEncoding[4];
	PGPError	err 				= kPGPError_NoErr;

	PGPValidatePtr( src );
	PGPValidateBERElementRef( ber );

	err = pgpBERGetLengthEncoding( srcLength, lengthEncoding, &lengthLength ); CKERR;

	err = pgpBERGetTagEncoding( tag, tagEncoding, &tagLength ); CKERR;

	length = ber->length + srcLength + tagLength + lengthLength;

	s = PGPNewData( ber->memMgr, length, kPGPMemoryMgrFlags_Clear );
	if( IsNull( s ) )
		return kPGPError_OutOfMemory;

	if( ber->length != 0 )
		pgpCopyMemory( ber->encoding, s, ber->length );
	
	pgpCopyMemory( tagEncoding, s + ber->length, tagLength );
	ber->length += tagLength;

	pgpCopyMemory( lengthEncoding, s + ber->length, lengthLength );
	ber->length += lengthLength;

	pgpCopyMemory( src, s + ber->length, srcLength );
	ber->length += srcLength;

	if( IsntNull( ber->encoding ) )
	{
		err = PGPFreeData( ber->encoding );	CKERR;
		ber->encoding = NULL;
	}

	ber->encoding = s;

error:
	return kPGPError_NoErr;
}

	PGPError
pgpSeekLastBERElement(
	PGPberElementRef 	ber, 
	PGPberElementRef *	lastBER )
{
	PGPberElementRef 	berPtr = NULL;
	PGPberSequence *	seqPtr = NULL;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( lastBER );

	/*
	 * Seek to the last PGPberElement in our chain of PGPberElements and
	 * PGPberSequences/PGPberSets.
	 */

	berPtr = ber;
	for( seqPtr = berPtr->next.nextSeq; seqPtr != NULL; seqPtr = berPtr->next.nextSeq )
	{
		berPtr = seqPtr->nextBER;

		/* berPtr should never be NULL */
		PGPValidatePtr( berPtr );
	}

	*lastBER = berPtr;

	return kPGPError_NoErr;
}


	PGPError
pgpBERBeginSequence(
	PGPberElementRef	ber,
	PGPUInt32			tag )
{
	PGPberSequence *	seq 	= NULL;
	PGPError			err 	= kPGPError_NoErr;
	PGPberElementRef 	lastBER = NULL;
	PGPberElementRef 	newBER	= NULL;

	PGPValidateBERElementRef( ber );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	/* Make new PGPberSequence */
	seq = PGPNewData( ber->memMgr, 
			  sizeof( PGPberSequence ), 
			  kPGPMemoryMgrFlags_Clear );
	if( IsNull( seq ) )
		return kPGPError_OutOfMemory;

	/* Make new PGPberElement */
	err = pgpNewBERElement( ber->memMgr, &newBER ); CKERR;

	newBER->tag = tag;

	/* Set pointers 
	 *
	 * This is how the pointer chain should look:
	 *
	 * [lastBER] -> [seq] -> [newBER]
	 */
	lastBER->next.nextSeq = seq;
	seq->prevBER = lastBER;

	seq->nextBER = newBER;
	newBER->prev.prevSeq = seq;

	goto done;
error:
	if( IsntNull( seq ) )
		(void) PGPFreeData( seq );

	if( !PGPberElementRefIsValid( newBER ) )
		(void) PGPFreeBERElement( newBER );

done:
	
	return err;
}

	PGPError
pgpBEREndSequence(
	PGPberElementRef	ber )
{
	PGPberElementRef	lastBER		= kInvalidPGPberElementRef;
	PGPberElementRef	newLastBER	= kInvalidPGPberElementRef;
	PGPberSequence *	seq			= NULL;
	PGPError			err 		= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	seq = lastBER->prev.prevSeq;
	newLastBER = seq->prevBER;

	err = pgpBERStrAppend( 
			newLastBER, 
			(IsNull( lastBER->encoding ) ? (PGPByte *) "": lastBER->encoding ), 
			lastBER->tag, 
			lastBER->length ); CKERR;

	/*
	 * We just ended a sequence... Now we need to free the last PGPberElement
	 * and fix the end of the pointer chain to indicate it's gone.
	 */
	(void) PGPFreeData( newLastBER->next.nextSeq );
	newLastBER->next.nextSeq = NULL;
	lastBER->prev.prevSeq = NULL;

	(void) PGPFreeBERElement( lastBER );

error:
	return err;
}

	PGPError
pgpBERBeginSet(
	PGPberElementRef	ber,
	PGPUInt32			tag )
{
	/*
	 * The only real difference between a sequence and a set is the tag, but
	 * this difference has already been accounted for, so we can just call
	 * pgpBERBeginSequence with a set tag.
	 */
	return pgpBERBeginSequence( ber, tag );
}

	PGPError
pgpBEREndSet(
	PGPberElementRef	ber )
{
	return pgpBEREndSequence( ber );
}

	PGPError
pgpBERInsertInt(
	PGPberElementRef	ber, 
	PGPInt32			value, 
	PGPUInt32			tag )
{
	PGPError			err 			= kPGPError_NoErr;
	PGPByte 			intString[16];
	PGPSize				intStringLength	= 0;
	PGPBoolean			sign			= FALSE;
	PGPInt32			i				= 0;
	PGPInt32			mask			= 0;
	PGPberElementRef	lastBER			= NULL;

	PGPValidateBERElementRef( ber );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	sign = (value < 0);

	/* Find the first byte that's not all ones or all zeros */
	for( i = sizeof( PGPInt32 ) - 1; i > 0; i-- )
	{
		mask = 0xFF << ( i * 8 );

		/* 
		 * if high bit is set, look for first non-all-one byte
		 * if high bit is not set, look for first non-all-zero byte
		 */
		if( sign )
		{
			if( ( value & mask ) != mask )	/* not all ones */
				break;
		}
		else
		{
			if( value & mask )		/* not all zeros */
				break;
		}
	}

	/*
	 * i now points to the first byte we encode.  If the high bit of
	 * this byte is not the same as the sign bit, when this is decoded,
	 * it will sign extend the wrong bit, so we need to tack an extra
	 * byte on to this to make sure it is sign extended correctly.
	 */
	mask = value & ( 0x80 << ( i * 8 ) );
	if( ( mask && !sign ) || ( !mask && sign ) )
		i++;

	/* Need to convert value to network byte order */
	PGPUInt32ToEndian( value, kPGPBigEndian, (PGPByte *) &value );

	intStringLength = i + 1;

	pgpCopyMemory( (char *) &value + sizeof( PGPInt32 ) - intStringLength, intString, intStringLength );

	err = pgpBERStrAppend( lastBER, intString, tag, intStringLength ); CKERR;

error:

	return err;
}

	PGPError
pgpBERInsertString(
	PGPberElementRef	ber,
	char *				string,
	PGPUInt32			tag )
{
	PGPError			err		= kPGPError_NoErr;
	PGPSize				length	= 0;
	PGPberElementRef	lastBER	= NULL;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( string );

	length = strlen( string );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	err = pgpBERStrAppend( lastBER, (PGPByte *) string, tag, length ); CKERR;

error:

	return err;
}

	PGPError
pgpBERInsertOctetstring(
	PGPberElementRef	ber,
	PGPByte *			string,
	PGPSize				length,
	PGPUInt32			tag )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	lastBER	= NULL;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( string );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	err = pgpBERStrAppend( lastBER, string, tag, length ); CKERR;

error:

	return err;
}

	PGPError
pgpBERInsertNULL(
	PGPberElementRef	ber,
	PGPUInt32			tag )
{
	PGPError			err		= kPGPError_NoErr;
	PGPberElementRef	lastBER	= kInvalidPGPberElementRef;

	PGPValidateBERElementRef( ber );

	err = pgpSeekLastBERElement( ber, &lastBER ); CKERR;

	err = pgpBERStrAppend( lastBER, (PGPByte *)"", tag, 0 ); CKERR;

error:

	return err;
}

	PGPError 
PGPberAppend(
	PGPberElementRef	ber, 
	const char *		fmt, 
	... )
{
	va_list 	argList;
	PGPUInt32	i					= 0;
	PGPUInt32	j					= 0;
	PGPError	err 				= kPGPError_NoErr;
	PGPUInt32	tag 				= kPGPberType_None;
	PGPBoolean	clearTag			= TRUE;

	char *			stringArg		= NULL;
	PGPByte *		octetStringArg	= NULL;
	char **			stringVectorArg	= NULL;
	PGPberValue **	berVectorArg	= NULL;
	PGPBoolean		boolArg			= FALSE;
	PGPInt32		intArg			= 0;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( fmt );

	va_start( argList, fmt );

	for( i = 0; fmt[i] != '\0'; i++ )
	{
		switch( fmt[i] )
		{
			case kPGPberFormatSpecifier_Boolean:
				boolArg = (PGPBoolean) va_arg( argList, PGPInt32 );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Boolean;

				/*
				 * Since the boolean will only be one byte long, we can pretend
				 * it's just an integer with a different tag (the boolean tag).
				 */
				err = pgpBERInsertInt( ber, boolArg, tag ); CKERR;
				break;

			case kPGPberFormatSpecifier_Int:
				intArg = va_arg( argList, PGPInt32 );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Int;

				err = pgpBERInsertInt( ber, intArg, tag ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_Octetstring:
				octetStringArg = va_arg( argList, PGPByte * );
				intArg = va_arg( argList, PGPSize );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Octetstring;

				err = pgpBERInsertOctetstring( ber, octetStringArg, intArg, tag ); CKERR;
				break;

			case kPGPberFormatSpecifier_String:
				stringArg = va_arg( argList, char * );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Octetstring;

				err = pgpBERInsertString( ber, stringArg, tag ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_StringVector:
				stringVectorArg = va_arg( argList, char ** );

				if( IsNull( stringVectorArg ) )
					break;

				if( tag == kPGPberType_None )
					tag = kPGPberType_Octetstring;

				for( j = 0; stringVectorArg[j] != NULL; j++ )
				{
					err = pgpBERInsertString( ber, stringVectorArg[j], tag );
					CKERR;
				}

				break;
			
			case kPGPberFormatSpecifier_BERVector:
				berVectorArg = va_arg( argList, PGPberValue ** );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Octetstring;

				for( j = 0; berVectorArg[j] != NULL; j++ )
				{
					err = pgpBERInsertOctetstring( ber, 
							berVectorArg[j]->value,
							berVectorArg[j]->length,
							tag	); CKERR;
				}

				break;
			
			case kPGPberFormatSpecifier_NULL:
				if( tag == kPGPberType_None )
					tag = kPGPberType_NULL;

				err = pgpBERInsertNULL( ber, tag ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_Enumeration:
				intArg = va_arg( argList, PGPInt32 );

				if( tag == kPGPberType_None )
					tag = kPGPberType_Enumeration;

				err = pgpBERInsertInt( ber, intArg, tag ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_Tag:
				tag = va_arg( argList, PGPUInt32 );
				clearTag = FALSE;
				break;

			case kPGPberFormatSpecifier_BeginSequence:
				if( tag == kPGPberType_None )
					tag = kPGPberType_Sequence;

				err = pgpBERBeginSequence( ber, tag ); CKERR;
				break;

			case kPGPberFormatSpecifier_EndSequence:
				err = pgpBEREndSequence( ber ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_BeginSet:
				if( tag == kPGPberType_None )
					tag = kPGPberType_Set;

				err = pgpBERBeginSet( ber, tag ); CKERR;
				break;
			
			case kPGPberFormatSpecifier_EndSet:
				err = pgpBEREndSet( ber ); CKERR;
				break;
			
			default:
				return kPGPError_BadParams;
				break;
		}

		/* If clearTag is set, we need to use a custom tag for the next
		 * element in the BER encoded string.  After we run through the above
		 * switch statement again, we have saved away the custom tag or used 
		 * it, and need to reset tag to the default.	*/

		if( clearTag )
			tag = kPGPberType_None;
		else
			clearTag = TRUE;
	}

	va_end( argList );

error:
	return err;
}



/****************************************************************************
 * BER decoding routines													*
 ****************************************************************************/

	PGPInt32
pgpReceiveLen(
	PGPSocketRef	sock,
	void *			buffer,
	PGPUInt32		bufferSize )
{
	PGPUInt32		bytesRead		= 0;
	PGPInt32		returnValue		= 0;

	PGPValidatePtr( buffer );

	while( bytesRead < bufferSize )
	{
		returnValue = PGPReceive( sock, ( (PGPByte *) buffer ) + bytesRead, 
							bufferSize - bytesRead,
							kPGPReceiveFlagNone );

		if( returnValue <= 0 )
			break;

		bytesRead += returnValue;
	}

	return bytesRead;
}

	PGPError
pgpBERReadTag(
	PGPByte	*			s,
	PGPberType *		tag,
	PGPSize *			tagLength)
{
	PGPberType			netTag		= kPGPberType_None;
	PGPByte				c			= 0;
	PGPUInt32			i			= 0;

	PGPValidatePtr( s );
	PGPValidatePtr( tag );
	PGPValidatePtr( tagLength );

	if( ( s[0] & kPGPberMask_BigTag ) == kPGPberMask_BigTag )
	{
		netTag = (PGPberType)0;
		i = 0;
		do
		{
			i++;
			c = s[i] & ~kPGPberMask_MoreTag;

			netTag >>= 8;
			pgpCopyMemory( &c, (PGPByte *) &netTag + ( sizeof( PGPberType ) - 1 ), 1 );

		} while( s[i] & kPGPberMask_MoreTag );


		*tag = PGPNetToHostLong( netTag );
		*tagLength = i + 1;
	}
	else
	{
		*tag = (PGPberType)s[0];
		*tagLength = 1;
	}

	return kPGPError_NoErr;
}

	PGPError
pgpBERReadUInt(
	PGPByte *			s,
	PGPSize				length,
	PGPUInt32 *			outInt )
{
	PGPError		err			= kPGPError_NoErr;

	PGPValidatePtr( s );
	PGPValidatePtr( outInt );
	
	*outInt = 0;

	switch( length )
	{
		case 1:
			*outInt = (PGPInt32) s[0];
			break;

		case 2:
			*outInt = (PGPUInt32) PGPEndianToUInt16( kPGPBigEndian, s );
			break;

		case 3:
			*outInt = (PGPUInt32) PGPEndianToUInt16( kPGPBigEndian, s );
			*outInt <<= 8;
			*outInt |= s[2];
			break;

		case 4:
			*outInt = PGPEndianToUInt32( kPGPBigEndian, s );
			break;

		default:
			err = kPGPError_BadParams;
			*outInt = 0;
			break;
	}

	return err;
}

	PGPError
pgpBERReadLength(
	PGPByte	*			s,
	PGPSize *			length ,
	PGPSize *			lengthOfLength )
{
	PGPByte		byteLen		= 0;
	PGPByte		lengthString[sizeof( PGPSize )];
	PGPError	err			= kPGPError_NoErr;
	PGPUInt32	i			= 0;

	PGPValidatePtr( s );
	PGPValidatePtr( length );
	PGPValidatePtr( lengthOfLength );

	byteLen = s[0];

	if( byteLen & kPGPberMask_BigLength )
	{
		*lengthOfLength = byteLen & ( ~kPGPberMask_BigLength );

		if( *lengthOfLength > 4 )
		{
			err = kPGPError_CorruptData;
			goto error;
		}

		pgpCopyMemory( s + 1, lengthString, *lengthOfLength );

		err = pgpBERReadUInt( lengthString, *lengthOfLength, (PGPUInt32 *)length ); CKERR;

		/* Add the length of byteLen */
		(*lengthOfLength)++;
	}
	else if( byteLen == kPGPberMask_IndefLength )
	{
		/*
		 * String of indefinite length.  This means that the length
		 * is given as 0 bytes, and we need to read the string until we
		 * find the end-of-contents octets (two zero octets).
		 */
		pgpAssert( 0 );		/* Untested */
		err = kPGPError_CorruptData;
		goto error;
		for( i = 1; ( s[i] != '\0' ) || ( s[i+1] != '\0' ); i++ )
			; /* NULL */

		*lengthOfLength = 1;
		*length = i + 1;
	}
	else
	{
		*lengthOfLength = 1;
		*length = byteLen;
	}

error:
	return err;
}

	PGPError
PGPberReadResponse(
	PGPberElementRef	ber,
	PGPSocketRef		sock )
{
	PGPByte		byteTag			= (PGPByte) kPGPberType_None;
	PGPByte		byteLen			= 0;
	PGPSize		len				= 0;
	PGPSize		lengthOfLength	= 0;
	PGPSize		prefixLength	= 0;
	PGPError	err				= kPGPError_NoErr;
	PGPByte		s[16];

	PGPValidateBERElementRef( ber );

	if( pgpReceiveLen( sock, &byteTag, 1 ) != 1 )
	{
		err = PGPGetLastSocketsError();
		goto error;
	}

	if( ( byteTag & kPGPberMask_BigTag ) == kPGPberMask_BigTag )
	{
		/* Untested */
		pgpAssert( 0 );
		err = kPGPError_CorruptData;
		goto error;
		/*
		PGPUInt32	netTag			= kPGPberType_None;
		PGPSize		tagLength		= 0;
		PGPByte		c				= 0;
		PGPUInt32	i				= 0;

		netTag = 0;
		s[0] = byteTag;
		i = 1;
		do
		{
			if( pgpReceiveLen( sock, &c, 1 ) != 1 )
			{
				err = PGPGetLastSocketsError();
				goto error;
			}
			s[i++] = c;
		} while( c & kPGPberMask_MoreTag );

		pgpBERReadTag( s, (PGPberType *)&tag, &tagLength );
		prefixLength += tagLength;
		*/
	}
	else
		prefixLength++;

	if( pgpReceiveLen( sock, &byteLen, 1 ) != 1 )
	{
		err = PGPGetLastSocketsError();
		goto error;
	}
	prefixLength++;

	if( byteLen & kPGPberMask_BigLength )
	{
		lengthOfLength = byteLen & ( ~kPGPberMask_BigLength );

		if( lengthOfLength > sizeof( s ) )
		{
			err = kPGPError_BadParams;
			goto error;
		}

		if( pgpReceiveLen( sock, s, (PGPUInt32) lengthOfLength ) != 
				(PGPInt32) lengthOfLength )
		{
			err = PGPGetLastSocketsError();
			goto error;
		}

		err = pgpBERReadUInt( s, lengthOfLength, (PGPUInt32 *)&len ); CKERR;

		prefixLength += lengthOfLength;
	}
	else
	{
		lengthOfLength = 1;
		len = byteLen;
	}


	ber->encoding = PGPNewData( ber->memMgr, len + prefixLength, 
						kPGPMemoryMgrFlags_Clear );
	if( IsNull( ber->encoding ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}
	
	/*pgpCopyMemory( &byteTag, ber->encoding, 1 ); */
	ber->encoding[0] = byteTag;

	if( byteLen & kPGPberMask_BigLength )
	{
		pgpCopyMemory( &byteLen, ber->encoding + 1, 1 );
		pgpCopyMemory( s, ber->encoding + 2, lengthOfLength );
	}
	else
		pgpCopyMemory( &byteLen, ber->encoding + 1, 1 );

	if( pgpReceiveLen( sock, ber->encoding + prefixLength, (PGPInt32) len ) 
			!= (PGPInt32) len )
	{
		err = PGPGetLastSocketsError();
		goto error;
	}

	ber->length = len + prefixLength;

	goto done;
error:

	if( IsntNull( ber ) )
	{
		if( IsntNull( ber->encoding ) )
		{
			(void) PGPFreeData( ber->encoding );
			ber->encoding = NULL;
		}
	}

	if( IsntPGPError( err ) )
		err = kPGPError_UnknownError;

done:

	return err;
}

	PGPError
PGPberSetData(
	PGPberElementRef	ber,
	PGPByte *			data,
	PGPSize				len )
{
	PGPError	err				= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	ber->encoding = PGPNewData( ber->memMgr, len, kPGPMemoryMgrFlags_Clear );
	if( IsNull( ber->encoding ) )
	{
		err = kPGPError_OutOfMemory;
		goto error;
	}

	pgpCopyMemory( data, ber->encoding, len );
	
	ber->length = len;

	goto done;
error:

	if( IsntNull( ber->encoding ) )
	{
		(void) PGPFreeData( ber->encoding );
		ber->encoding = NULL;
	}

	if( IsntPGPError( err ) )
		err = kPGPError_UnknownError;

done:

	return err;
}

	PGPError
pgpBERReadInt(
	PGPByte *			s,
	PGPSize				length,
	PGPInt32 *			outInt )
{
	PGPError		err			= kPGPError_NoErr;
	PGPUInt32		i			= 0;
	PGPInt32		sign		= 0;

	PGPValidatePtr( s );
	PGPValidatePtr( outInt );
	
	err = pgpBERReadUInt( s, length, (PGPUInt32 *) outInt ); CKERR;

	sign = *outInt & ( 0x80 << ( ( i - 1 ) * 8 ) );

	if( sign )
	{
		/* Sign extend */
		for( i = length; i < sizeof( PGPInt32 ); i++)
			*outInt |= ( 0xFF << ( i * 8 ) );
	}

error:
	return err;
}

	PGPError
pgpBERReadString(
	PGPByte *			s,
	PGPSize				length,
	PGPByte *			outString )
{
	PGPValidatePtr( s );
	PGPValidatePtr( outString );

	pgpCopyMemory( s, outString, length );

	return kPGPError_NoErr;
}

	PGPError
PGPberRewind(
	PGPberElementRef	ber )
{
	PGPValidatePtr( ber );

	return PGPberSetIndex( ber, 0 );
}

	PGPError
PGPberPeek(
	PGPberElementRef	ber,
	PGPberType *		tag,
	PGPSize *			len )
{
	PGPByte *			string		= NULL;
	PGPUInt32			unitLength	= 0;
	PGPError			err			= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( tag );
	PGPValidatePtr( len );

	if( ber->nextReadIndex == ber->length )
	{
		err = kPGPError_EndOfIteration;
		goto done;
	}

	string = ber->encoding;
	string += ber->nextReadIndex;

	err = pgpBERReadTag( string, tag, (PGPSize *)&unitLength ); CKERR;
	string += unitLength;

	err = pgpBERReadLength( string, len, (PGPSize *)&unitLength ); CKERR;
	string += unitLength;

error:
done:
	return err;
}

	PGPError
PGPberNextPrimitive(
	PGPberElementRef	ber )
{
	PGPldapType			tag				= kPGPldapType_None;
	PGPSize				len				= 0;
	PGPSize				unitLength		= 0;
	PGPByte *			string			= NULL;
	PGPError			err				= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	string = ber->encoding + ber->nextReadIndex;

	if( ber->nextReadIndex == ber->length )
	{
		err = kPGPError_EndOfIteration;
		goto done;
	}

	do
	{
		if( ber->nextReadIndex == ber->length )
		{
			err = kPGPError_NoErr;
			goto done;
		}

		ber->nextReadIndex = ( string - ber->encoding );

		err = pgpBERReadTag( string, (PGPberType *)&tag, &unitLength ); CKERR;
		string += unitLength;

		err = pgpBERReadLength( string, &len, &unitLength ); CKERR;
		string += unitLength;

	} while ( tag & kPGPberMask_Constructed );

error:
done:
	return err;
}

	PGPError
PGPberNextConstructed(
	PGPberElementRef	ber )
{
	PGPldapType			tag				= kPGPldapType_None;
	PGPSize				len				= 0;
	PGPSize				unitLength		= 0;
	PGPByte *			string			= NULL;
	PGPError			err				= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	string = ber->encoding + ber->nextReadIndex;

	if( ber->nextReadIndex == ber->length )
	{
		err = kPGPError_EndOfIteration;
		goto done;
	}

	do
	{
		if( ber->nextReadIndex == ber->length )
		{
			err = kPGPError_NoErr;
			goto done;
		}

		err = pgpBERReadTag( string, (PGPberType *)&tag, &unitLength ); CKERR;
		string += unitLength;

		err = pgpBERReadLength( string, &len, &unitLength ); CKERR;
		string += unitLength;

		if( !( tag & kPGPberMask_Constructed ) )	/* primitive */
			string += len;

		ber->nextReadIndex = ( string - ber->encoding );

	} while ( !( tag & kPGPberMask_Constructed ) );

error:
done:
	return err;
}


	PGPError
PGPberNext(
	PGPberElementRef	ber )
{
	PGPldapType			tag				= kPGPldapType_None;
	PGPSize				len				= 0;
	PGPSize				unitLength		= 0;
	PGPByte *			string			= NULL;
	PGPError			err				= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	string = ber->encoding + ber->nextReadIndex;

	if( ber->nextReadIndex == ber->length )
	{
		err = kPGPError_EndOfIteration;
		goto done;
	}

	err = pgpBERReadTag( string, (PGPberType *)&tag, &unitLength ); CKERR;
	string += unitLength;

	err = pgpBERReadLength( string, &len, &unitLength ); CKERR;
	string += unitLength;

	ber->nextReadIndex = ( string - ber->encoding );

error:
done:
	return err;
}


	PGPError
PGPberSkip(
	PGPberElementRef	ber )
{
	PGPldapType			tag				= kPGPldapType_None;
	PGPSize				len				= 0;
	PGPSize				unitLength		= 0;
	PGPByte *			string			= NULL;
	PGPError			err			= kPGPError_NoErr;

	PGPValidateBERElementRef( ber );

	if( ber->nextReadIndex == ber->length )
	{
		err = kPGPError_EndOfIteration;
		goto done;
	}

	string = ber->encoding + ber->nextReadIndex;

	err = pgpBERReadTag( string, (PGPberType *)&tag, &unitLength ); CKERR;
	string += unitLength;

	err = pgpBERReadLength( string, &len, &unitLength ); CKERR;
	string += unitLength;

	string += len;

	ber->nextReadIndex = ( string - ber->encoding );

error:
done:
	return err;
}

	PGPError
pgpBERFormatSpecifierToTag(
	PGPberFormatSpecifier		fmt,
	PGPberType *				berType )
{
	PGPError					err	= kPGPError_NoErr;

	PGPValidatePtr( berType );

	/* This conversion will only work with primitive BER types. */

	switch( fmt )
	{
		case kPGPberFormatSpecifier_Boolean:
			*berType = kPGPberType_Boolean;
			break;

		case kPGPberFormatSpecifier_Int:
			*berType = kPGPberType_Int;
			break;

		case kPGPberFormatSpecifier_Octetstring:
		case kPGPberFormatSpecifier_String:
			*berType = kPGPberType_Octetstring;
			break;

		case kPGPberFormatSpecifier_NULL:
			*berType = kPGPberType_NULL;
			break;

		case kPGPberFormatSpecifier_Enumeration:
			*berType = kPGPberType_Enumeration;
			break;

		default:
			*berType = kPGPberType_None;
			err = kPGPError_BadParams;
			break;
	}

	return err;
}

	PGPError
PGPberRead(
	PGPberElementRef	ber,
	const char *		fmt,
	... )
{
	va_list				argList;
	PGPUInt32			fmtIndex	= 0;
	PGPBoolean			nextIndex	= FALSE;
	PGPError			err 		= kPGPError_NoErr;
	PGPberType			tag			= kPGPberType_None;
	PGPberType			switchTag	= kPGPberType_None;
	PGPSize				length		= 0;
	PGPUInt32			unitLength	= 0;
	PGPByte	*			string		= NULL;

	PGPInt32 *			outInt		= NULL;
	PGPldapType *		outTag		= NULL;
	PGPByte *			outString	= NULL;
	PGPBoolean *		outBoolean	= NULL;

	PGPValidateBERElementRef( ber );
	PGPValidatePtr( fmt );

	va_start( argList, fmt );

	string = ber->encoding;
	string += ber->nextReadIndex;

	/* Run through the ber encoded string and look for elements */
	/*
	 * For each element, we need to look at:
	 * 1.	the tag
	 * 2.	the length
	 * 3.	the contents
	 */
	while(	( string < ( ber->encoding + ber->length ) ) && 
			( fmt[fmtIndex] != '\0' ) )
	{
		err = pgpBERReadTag( string, (PGPberType *)&tag, (PGPSize *)&unitLength ); CKERR;
		string += unitLength;

		err = pgpBERReadLength( string, &length, (PGPSize *)&unitLength ); CKERR;
		string += unitLength;

		if( length > (PGPUInt32)( ber->encoding + ber->length - string ) )
		{
			err = kPGPError_CorruptData;
			goto error;
		}

		switchTag = tag;
		if( fmt[fmtIndex] == kPGPberFormatSpecifier_Force )
		{
			/*
			 * We expect this to be a certain type, but the tag may not
			 * indicate the correct type.  We need to look at the format
			 * specifier and see what we think it should be.
			 */
			fmtIndex++;
			if( fmt[fmtIndex] == '\0' )
			{
				err = kPGPError_BadParams;
				CKERR;
			}

			err = pgpBERFormatSpecifierToTag( (PGPberFormatSpecifier)fmt[fmtIndex],
						&switchTag ); CKERR;
		}

		switch( switchTag )
		{
			case kPGPberType_Boolean:
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_Boolean )
				{
					nextIndex = TRUE;

					outBoolean = va_arg( argList, PGPBoolean * );
					err = pgpBERReadInt( string, length, (PGPInt32 *) outBoolean ); CKERR;
				}
				
				string += length;

				break;

			case kPGPberType_Int:
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_Int )
				{
					nextIndex = TRUE;

					outInt = va_arg( argList, PGPInt32 * );
					err = pgpBERReadInt( string, length, outInt ); CKERR;
				}

				string += length;

				break;

			case kPGPberType_Octetstring:
				if( ( fmt[fmtIndex] == kPGPberFormatSpecifier_String ) ||
					( fmt[fmtIndex] == kPGPberFormatSpecifier_Octetstring) )
				{
					nextIndex = TRUE;

					outString = va_arg( argList, PGPByte * );
					err = pgpBERReadString( string, length, outString ); CKERR;
				
					if( fmt[fmtIndex] == kPGPberFormatSpecifier_String )
						outString[length] = '\0';
				}

				string += length;

				break;

			case kPGPberType_NULL:
				/*
				 * There's really nothing to read, so we'll just skip the
				 * NULLs 
				 */
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_NULL )
					nextIndex = TRUE;

				string += length;

				break;

			case kPGPberType_Enumeration:
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_Enumeration )
				{
					nextIndex = TRUE;

					outInt = va_arg( argList, PGPInt32 * );
					err = pgpBERReadInt( string, length, outInt ); CKERR;
				}

				string += length;

				break;

			case kPGPberType_Sequence:
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_BeginSequence )
					nextIndex = TRUE;

				break;

			case kPGPberType_Set:
				if( fmt[fmtIndex] == kPGPberFormatSpecifier_BeginSet )
					nextIndex = TRUE;

				break;

			default:
				break;
		}

		if( fmt[fmtIndex] == kPGPberFormatSpecifier_Tag )
		{
			outTag = va_arg( argList, PGPldapType * );
			*outTag = (PGPldapType)tag;

			nextIndex = TRUE;
		}

		if( nextIndex )
			fmtIndex++;
		nextIndex = FALSE;
	}

	ber->nextReadIndex = ( string - ber->encoding );

	va_end( argList );

error:
	return err;
}

	PGPError
PGPberGetLength(
	PGPberElementRef	ber,
	PGPSize *			length )
{
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( length );

	*length = ber->length;

	return kPGPError_NoErr;
}

	PGPError
PGPberGetEncoding(
	PGPberElementRef	ber,
	PGPByte **			encoding )
{
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( encoding );

	*encoding = ber->encoding;

	return kPGPError_NoErr;
}

	PGPError
PGPberGetIndex(
	PGPberElementRef	ber,
	PGPUInt32 *			index )
{
	PGPValidateBERElementRef( ber );
	PGPValidatePtr( index );

	*index = ber->nextReadIndex;

	return kPGPError_NoErr;
}

	PGPError
PGPberSetIndex(
	PGPberElementRef	ber,
	PGPUInt32			index )
{
	PGPValidateBERElementRef( ber );

	ber->nextReadIndex = index;

	return kPGPError_NoErr;
}


/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
